from fastapi import APIRouter, Depends, HTTPException, status
from typing import Dict, Any
from fastapi.security import HTTPAuthorizationCredentials, HTTPBearer
from bson import ObjectId
from app.utility.security import verify_jwt
from app.db.db import get_col
from app.models.subadmin import SubAdmin, SubAdminLoginRequest, SubAdminLoginResponse
from datetime import datetime

router = APIRouter(prefix="/subadmins", tags=["Subadmins"])

admins_col = get_col("admins")
auth_scheme = HTTPBearer()

def _require_admin(credentials: HTTPAuthorizationCredentials = Depends(auth_scheme)) -> Dict[str, Any]:
    if not credentials:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="Not authenticated")
    ok, payload = verify_jwt(credentials.credentials)
    if not ok or not payload:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="Invalid token")
    if payload.get("role") != "admin":
        raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="admin access required")
    admin = admins_col.find_one({"_id": ObjectId(payload["sub"])})
    if not admin:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="admin not found")
    return admin

@router.post("/create/subadmin", response_model=SubAdmin, status_code=status.HTTP_201_CREATED)
def create_subadmin(payload: SubAdmin, admin=Depends(_require_admin)):
    now = datetime.utcnow()
    body = payload.dict()
    body["createdAt"] = now
    body["updatedAt"] = now
    res = admins_col.insert_one(body)
    created = admins_col.find_one({"_id": res.inserted_id})
    return {"message": "Subadmin created", "data": created}

@router.put("/update/subadmin/{subadmin_id}", response_model=SubAdmin, status_code=status.HTTP_200_OK)
def update_subadmin(subadmin_id: str, payload: SubAdmin, admin=Depends(_require_admin)):
    oid = ObjectId(subadmin_id)
    body = payload.dict()
    body["updatedAt"] = datetime.utcnow()
    res = admins_col.update_one({"_id": oid}, {"$set": body})
    if res.matched_count == 0:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Subadmin not found")
    updated = admins_col.find_one({"_id": oid})
    return {"message": "Subadmin updated", "data": updated}

@router.delete("/delete/subadmin/{subadmin_id}", status_code=status.HTTP_200_OK)
def delete_subadmin(subadmin_id: str, admin=Depends(_require_admin)):
    oid = ObjectId(subadmin_id)
    res = admins_col.delete_one({"_id": oid})
    if res.deleted_count == 0:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Subadmin not found")
    return {"message": "Subadmin deleted"}

@router.get("/get/subadmin/{subadmin_id}", response_model=SubAdmin, status_code=status.HTTP_200_OK)
def get_subadmin(subadmin_id: str, admin=Depends(_require_admin)):
    oid = ObjectId(subadmin_id)
    subadmin = admins_col.find_one({"_id": oid})
    if not subadmin:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Subadmin not found")
    return {"message": "Subadmin found", "data": subadmin}  
