from typing import List, Optional
from pydantic import BaseModel, Field
from bson import ObjectId
from datetime import datetime
# Helper to handle ObjectId conversion
class PyObjectId(ObjectId):
    @classmethod
    def __get_validators__(cls):
        yield cls.validate
    
    @classmethod
    def validate(cls, v):
        if not ObjectId.is_valid(v):
            raise ValueError("Invalid ObjectId")
        return ObjectId(v)

# --- Item Model ---
class CartItem(BaseModel):
    product: Optional[PyObjectId] = Field(None, alias="product")
    quantity: int = Field(..., gt=0)
    variantType: Optional[str] = None
    variantValue: Optional[str] = None
    price: Optional[float] = 0
    variantId: Optional[str] = None
    createdAt: datetime = Field(default_factory=datetime.utcnow)
    updatedAt: datetime = Field(default_factory=datetime.utcnow)
 

# --- Cart Model ---
class CartModel(BaseModel):
    user: PyObjectId = Field(..., alias="user")
    items: List[CartItem] = []
    totalAmount: float = 0
    platformFee: float = 8.00
    shippingCharges: float = 0
    finalAmount: float = 0
    createdAt: datetime = Field(default_factory=datetime.utcnow)
    updatedAt: datetime = Field(default_factory=datetime.utcnow)
    class Config:
        allow_population_by_field_name = True
        arbitrary_types_allowed = True
        json_encoders = {ObjectId: str}
